﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml.Linq;

namespace wol
{
    public class OwlObject
    {

		public const string DublinCoreDescription = "description";
		public const string OwlClass = "Class";
		public const string RdfAbout = "about";
		public const string RdfResource = "resource";
		public const string RdfDatatype = "datatype";
		public const string RdfsComment = "comment";
		public const string RdfsLabel = "label";
		public const string RdfType = "type";

		#region the OWL model that contains this object

		/// <summary>
		/// Gets and sets the model that has this object.
		/// </summary>
		/// <remarks>
		/// Side-effect: The setter sets the namespace of this object to the default namespace of the model.
		/// </remarks>
		public OwlModel InModel
		{
			get { return mInModel; }
			set {
				mInModel = value;
				if (null != mInModel)
				{
					NameSpace = mInModel.DefaultNamespace;
				}
			}
		}

		private OwlModel mInModel = null;

		#endregion

		public string NameSpace { get; set; }

        // rdfs:label element
        public string Label { get; set; }

        // rdf:about attribute may be #name or &owl;name
        public string About { get; set; }

        // dc:description element with attribute rdf:datatype="&xsd;string"
        public string Description { get; set; }

        // rdfs:comment element
        public string Comment { get; set; }

		public bool Anonymous { get; set; }

		#region serialization

		/// <summary>
		/// Populate this object from an xml element.
		/// </summary>
		/// <param name="source">the xml element from which to populate this object</param>
		public virtual void Deserialize(XElement source)
        {
            XAttribute aboutAttribute = XmlHelp.AttributeOf(source, OwlModel.RdfSpace, RdfAbout);
            if (null != aboutAttribute)
            {
				string oldAbout = About;
				About = OwlNameInSpace.NormalizeName(aboutAttribute.Value, InModel);
				if (null != InModel)
				{
					ChangedName(oldAbout);
					Anonymous = false;
				}
            }
            foreach (System.Xml.Linq.XElement element in source.Elements())
            {
                DeserializeInner(element); 
            }
        }

		/// <summary>
		/// Using knowledge about the internal structure of this object, decode the xml elements representing that structure.
		/// </summary>
		/// <param name="source">the xml element from which to populate this object</param>
		/// <returns>whether the source element contained information for this object</returns>
        protected virtual bool DeserializeInner(System.Xml.Linq.XElement source)
        {
            bool result = false;
			if (XmlHelp.ElementIs(source, OwlModel.RdfsSpace, RdfsLabel))
			{
				Label = source.Value;
				result = true;
			}
			else
			{
				if (XmlHelp.ElementIs(source, OwlModel.RdfsSpace, RdfsComment))
				{
					Comment = source.Value;
					result = true;
				}
				else
				{
					if (XmlHelp.ElementIs(source, OwlModel.DublinSpace, DublinCoreDescription))
					{
						Description = source.Value;
						result = true;
					}
				}
			}
            return result;
        }

		#endregion

		protected virtual void ChangedName(string oldName)
		{
		}

		public virtual string Display
        {
            get
            {
                string result = "";
                if (null != About)
                {
                    result += About;
                }
                if (null != Label)
                {
                    result += "(" + Label + ")";
                }
                if (null != Description)
                {
                    result += System.Environment.NewLine + "\tdescription:" + Description;
                }
                if (null != Comment)
                {
                    result += System.Environment.NewLine + "\tcomment:" + Comment;
                }
                return result;
            }
        }

		public virtual string DisplayName
		{
			get
			{
				string result = "";
				if (null != About)
				{
					result += About;
				}
				if (null != Label)
				{
					result += "(" + Label + ")";
				}
				return result;
			}
		}

		public virtual string DisplayNameOrAll
		{
			get
			{
				string result = "";
				if (Anonymous)
				{
					result = Display;
				}
				else
				{
					result = DisplayName;
				}
				return result;
			}
		}

		public virtual string ShortName
		{
			get
			{
				return OwlNameInSpace.ExtractName(About);
			}
		}

    }
}
