﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml.Linq;

namespace wol
{
	public class OwlModel
	{

		#region Assimilate an owl/rdf model.

		public const string OwlDatatypeProperty = "DatatypeProperty";
		public const string OwlNamedIndividual = "NamedIndividual";
		public const string OwlObjectProperty = "ObjectProperty";
		public const string Rdf = "RDF";
		public const string OwlThing = "Thing";

		/// <summary>
		/// Categorize the individuals, classes, and relations from an owl/rdf document object model.
		/// </summary>
		/// <param name="dom">the owl/rdf document object model to be added to this model</param>
		/// <remarks>
		/// This method has side effects of adding content to the following private categories of model parts:
		///		mNamespaces
		///		mObjectProperties
		///		mDatatypeProperties
		///		mClasses
		///		mIndividuals
		/// </remarks>
		public void AssimilateThis(XDocument dom)
		{
			XElement trunk = dom.Root;
			DetermineDefaultNamespace(trunk);
			foreach (XElement element in trunk.Elements())
			{
				if (XmlHelp.ElementIs(element, OwlSpace, OwlObjectProperty))
				{
					AssimilateObjectProperty(element);
				}
				else if (XmlHelp.ElementIs(element, OwlSpace, OwlDatatypeProperty))
				{
					AssimilateDatatypeProperty(element);
				}
				else if (XmlHelp.ElementIs(element, OwlSpace, OwlObject.OwlClass))
				{
					AssimilateOwlClass(element);
				}
				else if (XmlHelp.ElementIs(element, OwlSpace, OwlNamedIndividual))
				{
					AssimilateNamedIndividual(element);
				}
				else
				{
					AssimilateClassMember(element);
				}
			}
		}

		/// <summary>
		/// Assimilate an object property element.
		/// </summary>
		/// <param name="element">an xml element that represents an object property</param>
		/// <remarks>
		/// "owl:ObjectProperty" deserializes to an ObjectProperty instance,
		///	stored in mObjectProperties dictionary under its qualified object property name.
		///	Any rdf:about attribute provides the value of ObjectProperty.About.
		///	Any rdfs:label element provides the value of ObjectProperty.Label.
		///	Any rdfs:comment element provides the value of ObjectProperty.Comment.
		///	Any dc:description element provides the value of ObjectProperty.Description.
		///	Any rdfs:range element provides the vallue of ObjectProperty.Range.
		///	Any rdfs:subPropertyOf element provides the value of ObjectProperty.SubpropertyOf.
		///	Any rdf:type rdf:resource attribute sets ObjectProperty.Asymmetric or .Irreflexive to "true".
		///	Any owl:inverseOf element provides the value of ObjectProperty.InverseOf.
		/// </remarks>
		private void AssimilateObjectProperty(XElement element) {
			XAttribute aboutAttribute = XmlHelp.AttributeOf(element, RdfSpace, OwlObject.RdfAbout);
			if (null == aboutAttribute)
			{
				// Any anonymous object properties in the assimilated model are ignored.
			}
			else
			{
				string normalName = OwlNameInSpace.NormalizeName(aboutAttribute.Value, this);
				ObjectProperty objectProperty = FindOrMakeObjectProperty(normalName);
				objectProperty.Deserialize(element);
			}
		}

		/// <summary>
		/// Assimilate a data type property element.
		/// </summary>
		/// <param name="element">an xml element that represents a data type property</param>
		/// <remarks>
		/// "owl:DatatypeProperty" deserializes to an DatatypeProperty instance,
		///	stored in mDatatypeProperties dictionary under its qualified data type property name.
		///	Any rdf:about attribute provides the value of DatatypeProperty.About.
		///	Any rdfs:label element provides the value of DatatypeProperty.Label.
		///	Any rdfs:comment element provides the value of DatatypeProperty.Comment.
		///	Any dc:description element provides the value of DatatypeProperty.Description.
		///	Any rdfs:range element provides the vallue of DatatypeProperty.Range.
		///	Any rdfs:subPropertyOf element provides the value of DatatypeProperty.SubpropertyOf.
		/// </remarks>
		private void AssimilateDatatypeProperty(XElement element)
		{
			XAttribute aboutAttribute = XmlHelp.AttributeOf(element, RdfSpace, OwlObject.RdfAbout);
			if (null == aboutAttribute)
			{
				// Any anonymous datatype properties in the assimilated model are ignored.
			}
			else
			{
				string normalName = OwlNameInSpace.NormalizeName(aboutAttribute.Value, this);
				DatatypeProperty datatypeProperty = FindOrMakeDatatypeProperty(normalName);
				datatypeProperty.Deserialize(element);
			}
		}

		/// <summary>
		/// Assimilate an owl class element.
		/// </summary>
		/// <param name="element">an xml element that represents an owl class</param>
		/// <remarks>
		///	"owl:Class" derserializes to an OwlClass instance,
		///	stored in mClasses dictionary under the name in the rdf:about attribute, if the attribute is present,
		///	or under a numeric name if the attribute is absent.
		///	Any rdf:about attribute provides the value of DatatypeProperty.About.
		///	Any rdfs:label element provides the value of DatatypeProperty.Label.
		///	Any rdfs:comment element provides the value of DatatypeProperty.Comment.
		///	Any dc:description element provides the value of DatatypeProperty.Description.
		///	Any rdfs:subclassOf rdf:resource attribute adds to the value of OwlClass.SubclassOf.
		///		(This form is a relic of when multiple inheritance was supported in Protege.)
		///	Any owl:equivalentClass element provides an anonymous class as the content of OwlClass.EquivalentClass.
		///	Any owl:unionOf element with attribute rdf:parseType="Collection" provides a list of classes in OwlClass.UnionOf.
		///		The classes to be unified come from the rdf:description elements within the owl:unionOf element,
		///		which identify classes either by name in rdf:about attribute, or anonymously.
		///	Any owl:intersectionOf element with attribute rdf:parseType="Collection" provides a list of classes in OwlClass.IntersectionOf.
		///		The classes to be intersected come from the rdf:description elements within the owl:intersectionOf element,
		///		which identify classes either by name in rdf:about attribute, or anonymously.
		///	Any owl:onProperty rdf:resource attribute provides the value of OwlClass.RestrictProperty.
		///	Any owl:onClass rdf:resource attribute provides the value of OwlClass.RestrictClass.
		///	Any owl:onDataRange rdf:resource attribute provides the value of OwlClass.RestrictDataRange.
		///	Any owl:qualifiedCardinality element provides the value of OwlClass.QualifiedCardinality.
		///		(This implementation fails to associate the cardinality with the datatype named by the rdf:datatype attribute.)
		///	Any owl:maxQualifiedCardinality element provides the value of OwlClass.MaximumQualifiedCardinality.
		///		(This implementation fails to associate the cardinality with the datatype named by the rdf:datatype attribute.)
		///	Any owl:minQualifiedCardinality element provides the value of OwlClass.MinimumQualifiedCardinality.
		///		(This implementation fails to associate the cardinality with the datatype named by the rdf:datatype attribute.)
		///	Any owl:someValuesFrom rdf:resource attribute provides the value of OwlClass.SomeValuesFrom.
		///	Any owl:allValueFrom rdf:resource attribute provides the value of OwlClass.AllValuesFrom.
		///	Any owl:hasValue element provides the value of OwlClass.HasValue.
		///	Any owl:oneOf element with attribute rdf:parseType="Collection" provides a list of individuals in OwlClass.OneOf.
		///		The individuals come from the rdf:description elements within the owl:oneOf element,
		///		which identify individuals either by name in rdf:about attribute, or anonymously.
		///	Any owl:cardinality element provides the value of OwlClass.Cardinality.
		///		(This implementation fails to associate the cardinality with the datatype named by the rdf:datatype attribute.)
		///	Any owl:disjointWith rdf:resource attribute adds a class to the list in OwlClass.DisjointWith.
		/// </remarks>
		private void AssimilateOwlClass(XElement element)
		{
			XAttribute aboutAttribute = XmlHelp.AttributeOf(element, RdfSpace, OwlObject.RdfAbout);
			OwlClass owlClass = null;
			if (null == aboutAttribute)
			{
				owlClass = MakeAnonymousClass();
			}
			else
			{
				string normalName = OwlNameInSpace.NormalizeName(aboutAttribute.Value, this);
				owlClass = FindOrMakeClass(normalName);
			}
			owlClass.Deserialize(element);
		}

		/// <summary>
		/// Assimilate a named individual element.
		/// </summary>
		/// <param name="element">an xml element that represents a named individual</param>
		/// <remarks>
		/// "owl:NamedIndividual" deserializes to an OwlIndividual instance,
		///	stored in mIndividuals dictionary under the name in the rdf:about attribute, if the attribute is present,
		///	or under a numeric name if the attribute is absent.
		///	Any rdf:about attribute provides the value of DatatypeProperty.About.
		///	Any rdfs:label element provides the value of DatatypeProperty.Label.
		///	Any rdfs:comment element provides the value of DatatypeProperty.Comment.
		///	Any dc:description element provides the value of DatatypeProperty.Description.
		///	Any rdf:type rdf:resource attribute adds a class to the list in OwlIndividual.MemberOf.
		/// </remarks>
		private void AssimilateNamedIndividual(XElement element)
		{
			XAttribute aboutAttribute = XmlHelp.AttributeOf(element, RdfSpace, OwlObject.RdfAbout);
			OwlIndividual individual = null;
			if (null == aboutAttribute)
			{
				individual = MakeAnonymousIndividual();
			}
			else
			{
				string normalName = OwlNameInSpace.NormalizeName(aboutAttribute.Value, this);
				individual = FindOrMakeIndividual(normalName);
				individual.Deserialize(element);
			}
		}

		/// <summary>
		/// Assimilate an individual in a class.
		/// </summary>
		/// <param name="element">an element whose name is and owl class name or Thing</param>
		/// <remarks>
		/// The "about" attribute names an individual in the class.
		///	Elements that have other names than the categories of owl model parts have for their name the name of a class.
		///	If the class so named is not "Thing", this deserializes to an OwlIndividual instance,
		///	stored in mIndividuals dictionary under the name in the rdf:about attribute, if the attribute is present,
		///	or under a numeric name if the attribute is absent.
		///	The class named by the name of the element adds to the list in OwlIndividual.MemberOf.
		/// </remarks>
		private void AssimilateClassMember(XElement element)
		{
			string normalName = OwlNameInSpace.NormalizeName(element.Name.ToString(), this);
			OwlClass owlClass = FindOrMakeClass(normalName);
			XAttribute aboutAttribute = XmlHelp.AttributeOf(element, RdfSpace, OwlObject.RdfAbout);
			OwlIndividual individual = null;
			if (null == aboutAttribute)
			{
				individual = MakeAnonymousIndividual();
			}
			else
			{
				string normalIndividualName = OwlNameInSpace.NormalizeName(aboutAttribute.Value, this);
				individual = FindOrMakeIndividual(normalIndividualName);
			}
			if (false == normalName.EndsWith(OwlNameInSpace.NamespaceRightBracket + OwlThing))
			{
				if (null == individual.MemberOf)
				{
					individual.MemberOf = new List<OwlClass>();
				}
				individual.MemberOf.Add(owlClass);
			}
			individual.Deserialize(element);
		}

		#endregion

		#region Display this model as text.

		/// <summary>
		/// Display the content of this model as text.
		/// </summary>
		public string AsText
		{
			get
			{
				string result = "";
				foreach (ObjectProperty item in mObjectProperties.Values)
				{
					result += item.Display + System.Environment.NewLine;
				}
				foreach (DatatypeProperty item in mDatatypeProperties.Values)
				{
					result += item.Display + System.Environment.NewLine;
				}
				foreach (OwlClass item in mClasses.Values)
				{
					result += item.Display + System.Environment.NewLine;
				}
				foreach (OwlIndividual item in mIndividuals.Values)
				{
					result += item.Display + System.Environment.NewLine;
				}
				return result;
			}
		}

		#endregion

		#region Render this model as an xml schema.

		/// <summary>
		/// Render this model as an xml schema.
		/// </summary>
		/// <remarks>
		/// The xml schema produced by this accessor has two main parts.
		/// 1. A list of simple types enumerates the values possible for each semantic attribute
		///		by listing the names of the individuals in the owl class with the name of the semantic attribute.
		///		An attribute group lists the semantic attributes,
		///		which the SEDS schema makes eligible for use in a Semantics element in a SEDS instance.
		///	2. A list of complex types expands the structure of classes derived from the Subnet class of the ModelOfOperation class,
		///		which the SEDS schema makes eligible for use in a SEDS instance.
		/// </remarks>
		public string AsXsd
		{
			get
			{
				ClassifyModelParts();
				int indent = 0;
				string result = OpenSchemaTrunkElement();
				++indent;
				foreach (string typeName in mTypesAndMembers.Keys)
				{
					if (typeName == "NotSemantic") continue;
					KeyValuePair<OwlClass, List<OwlIndividual>> pair = mTypesAndMembers[typeName];
					OwlClass item = pair.Key;
					if (null != IsOrDerivesFromImmediately(item, "ExtensibleEnumeration"))
					{
						result += GenerateEnumerationSimpleType(typeName, item, pair.Value, indent);
						continue;
					}
					if (null != IsOrDerivesFromImmediately(item, "SchemaProperty")) continue;
					// If class is semantic property, generate simple type with individuals enumerated.
					result += GenerateEnumerationSimpleType(typeName, item, pair.Value, indent);
				}
				if (null != mEnumerationClass)
				{
					result += GenerateEnumerationOfEnumerations(mEnumerationClass, mEnumClassNames, indent);
				}
				result += GenerateFlatModelOfOperations(indent);
				result += GenerateSedsCoreSemanticsAttributeGroup(mEnumerationClass, indent);
				// The following commented line generates complex types for the model of operation.
				// That is out-of-style at the present time, but might return after some experience with simple attributes.
				// result += GenerateModelOfOperations(indent);
				// The above commented line generates complex types for the model of operation.
				--indent;
				System.Diagnostics.Debug.Assert(0 == indent, "indentation error: " + indent.ToString() + ".");
				result += CloseElement("xs:schema", 0);
				return result;
			}
		}

		/// <summary>
		/// Generate simple types for the model of operation.
		/// </summary>
		/// <param name="indent">the amount of indentation for lines generated</param>
		/// <returns>xsd lines that define simple types for the model of operation</returns>
		private string GenerateFlatModelOfOperations(int indent)
		{
			string result = "";
			OwlClass specificValues = (from OwlClass item in mClasses.Values
									   where (item.ShortName == "SpecificValue")
									   select item).First();
			List<OwlIndividual> patterns = (from OwlIndividual individual in mIndividuals.Values
											where ((null != individual.MemberOf) && (individual.MemberOf.Count == 1)
											&& (individual.MemberOf[0].SubclassOf != null)
											&& (individual.MemberOf[0].SubclassOf.Contains(specificValues)))
											orderby individual.ShortName
											select individual).ToList();
			foreach(OwlIndividual model in patterns)
			{
				result += GeneratePatternSimpleType(model, indent);
				mSpecificValues.Add(model.ShortName, model);
			}
			foreach (OwlClass model in mModelsOfOps.Values)
			{
				if (ClassIsEnum(model))
				{
					// This class represents an enumeration.
					List<OwlIndividual> members = (from OwlIndividual member in mIndividuals.Values
												   where ((null != member.MemberOf) && (member.MemberOf.Contains(model)))
												   orderby member.ShortName
												   select member).ToList();
					result += GenerateEnumerationSimpleType(model.ShortName, model, members, indent);
				}
			}
			return result;
		}

		/// <summary>
		/// Generate complex types for the model of operation.
		/// </summary>
		/// <param name="indent">the amount of indentation for lines generated</param>
		/// <returns>xsd lines that define complex types for the model of operation</returns>
		private string GenerateModelOfOperations(int indent)
		{
			string result = "";
			foreach (OwlClass model in mModelsOfOps.Values)
			{
				if (ClassIsEnum(model))
				{
					// This class represents an enumeration.
					List<OwlIndividual> members = (from OwlIndividual member in mIndividuals.Values
												   where ((null != member.MemberOf) && (member.MemberOf.Contains(model)))
												   orderby member.ShortName
												   select member).ToList();
					result += GenerateEnumerationSimpleType(model.ShortName, model, members, indent);
				}
				else
				{
					// This class is not an enumeration.
					result += GenerateComplexType(model, indent);
				}
			}
			return result;
		}

		/// <summary>
		/// Generate an xml string that opens the trunk element of the schema.
		/// </summary>
		/// <returns>an xml string that opens the trunk element of the schema</returns>
		private string OpenSchemaTrunkElement()
		{
			string result = "<xs:schema targetNamespace=\"http://www.ccsds.org/schema/sois/seds\"";
			result += System.Environment.NewLine + "\txmlns:xs=\"http://www.w3.org/2001/XMLSchema\"";
			result += System.Environment.NewLine + "\txmlns:seds=\"http://www.ccsds.org/schema/sois/seds\"";
			result += System.Environment.NewLine + "\telementFormDefault=\"qualified\"";
			result += System.Environment.NewLine + ">";
			return result;
		}

		/// <summary>
		/// Generate a string that starts a line with an indentation.
		/// </summary>
		/// <param name="indent">the depth of the indentation</param>
		/// <returns>a string that starts a line with an indentation</returns>
		private string GenerateIndentation(int indent)
		{
			string result = System.Environment.NewLine;
			for (int ix = 0; ix < indent; ++ix) result += "\t";
			return result;
		}

		/// <summary>
		/// Generate an xml string that closes an element of the schema.
		/// </summary>
		/// <param name="name">the name of the element to be closed</param>
		/// <param name="indent">the minimum number of tabs to prefix each xml line</param>
		/// <returns>an xml string that closes an element of the schema</returns>
		private string CloseElement(string name, int indent)
		{
			string result = GenerateIndentation(indent) + "</" + name + ">";
			return result;
		}

		/// <summary>
		/// Generate an xml string that defines a simple type restricted by regular expression in a schema.
		/// </summary>
		/// <param name="specificValue">the owl class that defines the restriction</param>
		/// <param name="indent">the minimum nmber of tabs to prefix each xml line</param>
		/// <returns></returns>
		private string GeneratePatternSimpleType(OwlIndividual specificValue, int indent)
		{
			string result = OpenSimpleTypeElement(specificValue.ShortName, indent);
			++indent;
			if (null != specificValue)
			{
				result += GenerateAnnotation(specificValue.Description, indent);
			}
			result += OpenRestrictionElement("xs:string", indent);
			++indent;
			result += GenerateRegexElement(specificValue.Syntax, null, indent);
			--indent;
			result += CloseElement("xs:restriction", indent);
			--indent;
			result += CloseElement("xs:simpleType", indent);
			return result;
		}

		/// <summary>
		/// Generate an xml string that defines an enumeration in a schema.
		/// </summary>
		/// <param name="name">the name of the enumeration</param>
		/// <param name="enumerationSet">the class that defines the enumeration in this model</param>
		/// <param name="enumerationMembers">the individuals in the class that defines the enumeration</param>
		/// <param name="indent">the minimum number of tabs to prefix each xml line</param>
		/// <returns>an xml string that defines an enumeration in a schema</returns>
		private string GenerateEnumerationSimpleType(string name, OwlClass enumerationSet,
			List<OwlIndividual> enumerationMembers, int indent)
		{
			string result = OpenSimpleTypeElement(name, indent);
			++indent;
			if (null != enumerationSet)
			{
				result += GenerateAnnotation(enumerationSet.Description, indent);
			}
			result += OpenRestrictionElement("xs:string", indent);
			++indent;
			foreach (OwlIndividual member in enumerationMembers)
			{
				result += GenerateEnumerationElement(member.ShortName, member.Description, indent);
			}
			--indent;
			result += CloseElement("xs:restriction", indent);
			--indent;
			result += CloseElement("xs:simpleType", indent);
			return result;
		}

		/// <summary>
		/// Generate an xml string that defines a type of element that may contain other elements or attributes.
		/// </summary>
		/// <param name="model">a class in a model of operations</param>
		/// <param name="indent">the minimum number of tabs to prefix each xml line</param>
		/// <returns>an xml string that defines a type of element that may contain other elements or attributes</returns>
		private string GenerateComplexType(OwlClass model, int indent)
		{
			string result = OpenComplexTypeElement(model.ShortName, indent);
			++indent;
			GenerateAnnotation(model.Description, indent);
			--indent;
			if (null == model.EquivalentClass)
			{
				// This class has no object relations.
				result += GenerateBareComplexContent(model, indent);
			}
			else {
				//This class has object relations.
				++indent;
				result += GenerateComplexContent(model, indent);
				--indent;
			}
			result += CloseElement("xs:complexType", indent);
			return result;
		}

		/// <summary>
		/// Generate an xml string that defines a type of element that contains no elements or attributes.
		/// </summary>
		/// <param name="model">a class in a model of operations</param>
		/// <param name="indent">the minimum number of tabs to prefix each xml line</param>
		/// <returns>an xml string that defines a type of element that contains no elements or attributes</returns>
		private string GenerateBareComplexContent(OwlClass model, int indent)
		{
			string result = "";
			OwlClass superClass = model.SuperClass;
			if (null != superClass)
			{
				++indent;
				result += OpenElement("xs:complexContent", indent);
				++indent;
				result += OpenExtensionElement("seds:" + superClass.ShortName, indent);
			}
			++indent;
			//result += GenerateEmptySequence(indent);
			result += LicenseSemanticAttributes(model, indent);
			--indent;
			if (null != superClass)
			{
				result += CloseElement("xs:extension", indent);
				--indent;
				result += CloseElement("xs:complexContent", indent);
				--indent;
			}
			return result;
		}

		/// <summary>
		/// Generate an xml string that defines a type of element that may contain elements and attributes.
		/// </summary>
		/// <param name="model">a class in a model of operations</param>
		/// <param name="indent">the minimum number of tabs to prefix each xml line</param>
		/// <returns>an xml string that defines a type of element that may contain elements and attributes</returns>
		private string GenerateComplexContent(OwlClass model, int indent)
		{
			string result = OpenElement("xs:complexContent", indent);
			OwlClass superClass = model.SuperClass;
			if (null != superClass)
			{
				++indent;
				result += OpenExtensionElement("seds:" + superClass.ShortName, indent);
			}
			++indent;
			result += GenerateElementsForComplexType(model, indent);
			result += GenerateAttributesForComplexType(model, indent);
			result += LicenseSemanticAttributes(model, indent);
			--indent;
			if (null != superClass)
			{
				result += CloseElement("xs:extension", indent);
				--indent;
			}
			result += CloseElement("xs:complexContent", indent);
			return result;
		}

		/// <summary>
		/// Generate an xml string that defines the elements of a complex type.
		/// </summary>
		/// <param name="model">a class in a model of operations that represents the complex type</param>
		/// <param name="indent">the minimum number of tabs to prefix each xml line</param>
		/// <returns>an xml string that defines the elements of a complex type</returns>
		private string GenerateElementsForComplexType(OwlClass model, int indent)
		{
			string result = "";
			IEnumerable <OwlClass> restricteds = from OwlClass equivalent in model.EquivalentClass
												where (
												(equivalent.RestrictProperty != null) &&
												(equivalent.RelatedClass != null) &&
												(!ClassIsEnum(equivalent.RelatedClass))
												)
												orderby equivalent.RestrictProperty.ShortName
												select equivalent;
			if (restricteds.Count() > 0)
			{
				result += OpenElement("xs:sequence", indent);
                foreach (OwlClass equivalent in restricteds)
                {
                    Property relation = equivalent.RestrictProperty;
                    OwlClass related = equivalent.RelatedClass;
                    IEnumerable<OwlClass> derivatives = from OwlClass derivative in mClasses.Values
                                                        where (
                                                        (derivative.SuperClass != null) &&
                                                        (derivative.SuperClass.ShortName == related.ShortName)
                                                        )
                                                        select derivative;
                    if (0 == derivatives.Count())
                    {
                        ++indent;
                        result += GenerateRelationElement(relation.ShortName, "seds:" + related.ShortName, indent);
                        --indent;
                    } else
                    {
                        ++indent;
                        result += OpenChoiceElement("unbounded", indent);
                        ++indent;
                        foreach (OwlClass derivative in derivatives)
                        {
                            result += GenerateRelationElement(derivative.ShortName, "seds:" + derivative.ShortName, indent);
                        }
                        --indent;
                        result += CloseElement("xs:choice", indent);
                        --indent;
                    }
                }
				result += CloseElement("xs:sequence", indent);
			}
			return result;
		}

		/// <summary>
		/// Generate an xml string the defines the attributes of a complex type.
		/// </summary>
		/// <param name="model">a class in a model of operations that represents the complex type</param>
		/// <param name="indent">the minimum number of tabs to prefix each xml line</param>
		/// <returns>an xml string the defines the attributes of a complex type</returns>
		private string GenerateAttributesForComplexType(OwlClass model, int indent)
		{
			string result = "";
			List<string> attributeNamesUsed = new List<string>();
			IEnumerable<OwlClass> attributeModel = from OwlClass equivalent in model.EquivalentClass
												   where equivalent.RestrictProperty != null
												   orderby equivalent.RestrictProperty.ShortName
												   select equivalent;
			foreach (OwlClass equivalent in attributeModel)
			{
				Property relation = equivalent.RestrictProperty;
				System.Diagnostics.Debug.Assert(null != relation);
				{
					OwlClass related = equivalent.RelatedClass;
					if (null != related)
					{
						if (ClassIsEnum(related))
						{
							result += GenerateAttributeForComplexType(relation.ShortName,
								"seds:" + related.ShortName, attributeNamesUsed, indent);
						}
					}
					else
					{
						Datatype range = equivalent.RestrictDataRange;
						if (null != range)
						{
							string rangeName = range.ShortName;
							if (rangeName != "positiveInteger")//todo: Replace with constraints.
							{
								result += GenerateAttributeForComplexType(relation.ShortName, "xs:" + rangeName,
									attributeNamesUsed, indent);
							}
						}
					}
				}
			}
			return result;
		}

		/// <summary>
		/// Generate an xml string the defines an attribute of a complex type.
		/// </summary>
		/// <param name="attributeName">the name of the attribute</param>
		/// <param name="typeName">the type of the attribute</param>
		/// <param name="attributeNamesUsed">a list of attributes already defined in the complex type</param>
		/// <param name="indent">the minimum number of tabs to prefix each xml line</param>
		/// <returns>an xml string the defines an attribute of a complex type</returns>
		private string GenerateAttributeForComplexType(string attributeName, string typeName,
			List<string> attributeNamesUsed, int indent)
		{
			string result = "";
			if (!attributeNamesUsed.Contains(attributeName))
			{
				attributeNamesUsed.Add(attributeName);
				result += GenerateAttribute(attributeName, typeName, indent);
			}
			return result;
		}

		/// <summary>
		/// Generate an xml string that defines an enumeration of the names of enumerations in a schema.
		/// </summary>
		/// <param name="enumerationSet">the class that defines the enumeration in this model</param>
		/// <param name="enumerationMembers">the classes that are enumerations (except the enumerationSet)</param>
		/// <param name="indent">the minimum number of tabs to prefix each xml line</param>
		/// <returns>an xml string that defines an enumeration in a schema</returns>
		private string GenerateEnumerationOfEnumerations(OwlClass enumerationSet,
			Dictionary<string, OwlClass> enumerationMembers, int indent)
		{
			string result = OpenSimpleTypeElement("enumerationNames", indent);
			++indent;
			if (null != enumerationSet)
			{
				result += GenerateAnnotation(enumerationSet.Description, indent);
			}
			result += OpenRestrictionElement("xs:string", indent);
			++indent;
			foreach (OwlClass member in enumerationMembers.Values)
			{
				result += GenerateEnumerationElement(member.ShortName, member.Description, indent);
			}
			--indent;
			result += CloseElement("xs:restriction", indent);
			--indent;
			result += CloseElement("xs:simpleType", indent);
			return result;
		}

		/// <summary>
		/// Generate an xml string that declares the seds core semantics attribute group.
		/// </summary>
		/// <param name="enumerationClass">the class of enumerations, or null if none such</param>
		/// <param name="indent">the minimum number of tabs to prefix each xml line</param>
		/// <returns>an xml string that declares the seds core semantics attribute group</returns>
		private string GenerateSedsCoreSemanticsAttributeGroup(OwlClass enumerationClass, int indent){
			string result = OpenCoreSemanticsAttributeGroup(indent);
			++indent;
			foreach (string typeName in mTypesAndMembers.Keys)
			{
				if (typeName == "NotSemantic") continue;
				KeyValuePair<OwlClass, List<OwlIndividual>> pair = mTypesAndMembers[typeName];
				OwlClass item = pair.Key;
				if (null != IsOrDerivesFromImmediately(item, "SchemaProperty")) continue;
				if (null != IsOrDerivesFromImmediately(item, "ExtensibleEnumeration")) continue;
				result += GenerateAttribute(typeName, "seds:" + typeName, indent);
			}
			foreach (string typeName in mTypesWithoutMembers.Keys)
			{
				OwlClass item = mTypesWithoutMembers[typeName];
				if (null != IsOrDerivesFromImmediately(item, "SchemaProperty")) continue;
				result += GenerateAttribute(typeName, "xs:string", indent);
			}
			if (null != enumerationClass)
			{
				result += GenerateAttribute("enumeration", "seds:enumerationNames", indent);
			}
			foreach (OwlClass model in mModelsOfOps.Values)
			{
				if (ClassIsEnum(model))
				{
					// This class represents an enumeration.
					result += GenerateAttribute(model.ShortName, "seds:" + model.ShortName, indent);
				}
			}
			foreach (string typeName in mSpecificValues.Keys)
			{
				result += GenerateAttribute(typeName, "seds:" + mSpecificValues[typeName].ShortName, indent);
			}
			--indent;
			result += CloseElement("xs:attributeGroup", indent);
			return result;
		}

        /// <summary>
        /// Generate an xml string that represents an empty sequence.
        /// </summary>
        /// <param name="indent">the minimum number of tabs to prefix each xml line</param>
        /// <returns>an xml string that represents an empty sequence</returns>
        private string GenerateEmptySequence(int indent)
        {
			string result = GenerateIndentation(indent);
            result += "<xs:sequence/>";
            return result;
        }

        /// <summary>
        /// Generate an xml string that opens a simple type declaration in the schema.
        /// </summary>
        /// <param name="name">the name of the simple type</param>
        /// <param name="indent">the minimum number of tabs to prefix each xml line</param>
        /// <returns>an xml string that opens a simple type declaration in the schema</returns>
        private string OpenSimpleTypeElement(string name, int indent)
		{
			string result = GenerateIndentation(indent);
			result += "<xs:simpleType name=\"" + name + "\">";
			return result;
		}

        /// <summary>
        /// Generate an xml string that opens a complex type declaration in the schema.
        /// </summary>
        /// <param name="name">the name of the complex type</param>
        /// <param name="indent">the minimum number of tabs to prefix each xml line</param>
        /// <returns>an xml string that opens a complex type declaration in the schema</returns>
        private string OpenComplexTypeElement(string name, int indent)
        {
            string result = GenerateIndentation(indent);
			result += "<xs:complexType name=\"" + name + "\"";
			if (name == "ModelOfOperation")
			{
				result += " abstract=\"true\"";
			}
			result += ">";
            return result;
        }

        /// <summary>
        /// Generate an xml string that opens an element that will contain other elements.
        /// </summary>
        /// <param name="indent">the minimum number of tabs to prefix each xml line</param>
        /// <returns>an xml string that opens an element that will contain other elements</returns>
        private string OpenElement(string name, int indent)
        {
            string result = GenerateIndentation(indent);
            result += "<" + name + ">";
            return result;
        }

        /// <summary>
        /// Generate an xml string that opens a choice element.
        /// </summary>
        /// <param name="maxOccurs">the upper bound on the number of choices</param>
        /// <param name="indent">the minimum number of tabs to prefix each xml line</param>
        /// <returns>an xml string that opens a choice element</returns>
        private string OpenChoiceElement(string maxOccurs, int indent)
        {
            string result = GenerateIndentation(indent);
            result += "<xs:choice";
			if ("1" != maxOccurs)
			{
				result += " maxOccurs =\"" + maxOccurs + "\"";
			}
            result += ">";
            return result;
        }

        /// <summary>
        /// Generate an xml string that opens a restriction element.
        /// </summary>
        /// <param name="baseName">the name of the base type of the restriction</param>
        /// <param name="indent">the minimum number of tabs to prefix each xml line</param>
        /// <returns>an xml string that opens a restriction element</returns>
        private string OpenRestrictionElement(string baseName, int indent)
		{
			string result = GenerateIndentation(indent);
			result += "<xs:restriction base=\"" + baseName + "\">";
			return result;
		}

		/// <summary>
		/// Generate an xml string that defines a relation element.
		/// </summary>
		/// <param name="relationName">the name of the relation</param>
		/// <param name="relatedName">the related class</param>
		/// <param name="indent">the minimum number of tabs to prefix each xml line</param>
		/// <returns>an xml string that defines a relation element</returns>
		private string GenerateRelationElement(string relationName, string relatedName, int indent)
		{
			string result = GenerateIndentation(indent);
			result += "<xs:element name=\"" + relationName + "\" type=\"" + relatedName + "\"/>";
			return result;
		}

		/// <summary>
		/// Generate an xml string that opens an extension element.
		/// </summary>
		/// <param name="baseName">the name of the base type of the extension</param>
		/// <param name="indent">the minimum number of tabs to prefix each xml line</param>
		/// <returns>an xml string that opens an extension element</returns>
		private string OpenExtensionElement(string baseName, int indent)
        {
            string result = GenerateIndentation(indent);
            result += "<xs:extension base=\"" + baseName + "\">";
            return result;
        }

		/// <summary>
		/// Generate an xml string for a regular expression constraint in an xml schema.
		/// </summary>
		/// <param name="regex">the regular expression</param>
		/// <param name="description">a description of the regular expression</param>
		/// <param name="indent">the minimum number of tabs to preix each xml line</param>
		/// <returns></returns>
		private string GenerateRegexElement(string regex, string description, int indent)
		{
			string result = GenerateIndentation(indent);
			result += "<xs:pattern value=\"" + regex + "\"";
			if (null == description)
			{
				result += "/>";
			}
			else
			{
				result += ">";
				++indent;
				result += GenerateAnnotation(description, indent);
				--indent;
				result += CloseElement("xs:pattern", indent);
			}
			return result;
		}

        /// <summary>
        /// Generate an xml string for an enumeration element.
        /// </summary>
        /// <param name="enumerationLabel">the string that labels the enumeration item</param>
        /// <param name="description">a description of the enumeration item, or null</param>
        /// <param name="indent">the minimum number of tabs to prefix each xml line</param>
        /// <returns>an xml string that opens an enumeration element</returns>
        private string GenerateEnumerationElement(string enumerationLabel, string description, int indent)
		{
			string result = GenerateIndentation(indent);
			result += "<xs:enumeration value=\"" + enumerationLabel + "\"";
			if (null == description)
			{
				result += "/>";
			}
			else
			{
				result += ">";
				++indent;
				result += GenerateAnnotation(description, indent);
				--indent;
				result += CloseElement("xs:enumeration", indent);
			}
			return result;
		}

		/// <summary>
		/// Return an xml string for an annotation element.
		/// </summary>
		/// <param name="description">a string for the documentation element within the annotation element</param>
		/// <param name="indent">the minimum number of tabs to prefix each xml line</param>
		/// <returns>an xml string for an annotation element</returns>
		private string GenerateAnnotation(string description, int indent)
        {
            string result = "";
            if (null != description)
            {
				result += GenerateIndentation(indent);
                result += "<xs:annotation>";
                ++indent;
				result += GenerateIndentation(indent);
                result += "<xs:documentation>" + NormalizeLineEndsToWindows(description) + "</xs:documentation>";
                --indent;
				result += CloseElement("xs:annotation", indent);
            }
            return result;
        }

		/// <summary>
		/// Generate an xml string that opens the core semantics attribute group.
		/// </summary>
		/// <param name="indent">the minimum number of tabs to prefix each xml line</param>
		/// <returns>an xml string that opens the core semantics attribute group</returns>
		private string OpenCoreSemanticsAttributeGroup(int indent)
		{
			string result = GenerateIndentation(indent);
			result += "<xs:attributeGroup name=\"CoreSemanticsAttributeGroup\">";
			return result;
		}

		/// <summary>
		/// Generate an xml string that declares an attribute in a group.
		/// </summary>
		/// <param name="name">the name of the attribute</param>
		/// <param name="typeName">the type of the attribute</param>
		/// <param name="indent">the minimum number of tabs to prefix each xml line</param>
		/// <returns>an xml string that declares an attribute in a group</returns>
		private string GenerateAttribute(string name, string typeName, int indent)
		{
			string result = GenerateIndentation(indent);
			result += "<xs:attribute name=\"" + AsAttributeName(name) + "\" type=\"" + typeName + "\"/>";
			return result;
		}

        /// <summary>
        /// Assure that the first character of an attribute name is lower case.
        /// </summary>
        /// <param name="typeName">the attribute name to be adjusted</param>
        /// <returns>the attribute name with lower case first character</returns>
        private string AsAttributeName(string typeName)
        {
            string result = typeName;
            if (result.Substring(0, 1).ToUpper() == result.Substring(0, 1))
            {
                result = result.Substring(0, 1).ToLower() + result.Substring(1);
            }
            return result;
        }

        /// <summary>
        /// Generate an xml string that authorizes the use of the core semantics attribute group.
        /// </summary>
        /// <param name="model">the class whose schema is being generated</param>
        /// <param name="indent">the minimum number of tabs to prefix each xml line</param>
        /// <returns>an xml string that authorizes the use of the core semantics attribute group, or the empty string if unnecessary</returns>
        private string LicenseSemanticAttributes(OwlClass model, int indent)
        {
            string result = "";
            if (model.ShortName == "ModelOfOperation")
            {
                result += GenerateIndentation(indent);
                result += "<xs:attributeGroup ref=\"seds:CoreSemanticsAttributeGroup\"/>";
            }
            return result;
        }

		/// <summary>
		/// Determine whether a class represents an enumeration for a semantic attribute.
		/// </summary>
		/// <param name="toTest">the class whose enumeration representation is to be determined</param>
		/// <returns>whether the class represents an enumeration for a semantic attribute</returns>
		private bool ClassIsEnum(OwlClass toTest)
		{
			bool result = false;
			IEnumerable<OwlIndividual> individuals = from OwlIndividual individual in mIndividuals.Values
													 where (null != individual.MemberOf)
													 select individual;
			string toTestName = toTest.ShortName;
			foreach (OwlIndividual item in individuals)
			{
				foreach (OwlClass inSet in item.MemberOf)
				{
					if (inSet.ShortName == toTestName)
					{
						result = true;
						break;
					}
				}
				if (result) break;
			}
			return result;
		}

		#endregion

		#region Classify parts of the model for representation in schema or html

		private void ClassifyModelParts()
		{
			if (!mModelPartsHaveBeenClassified)
			{
				ClassifyIndividuals();
				IdentifyTypesWithoutMembers();
				IdentifyPartsOfModelOfOperations();
			}
			mModelPartsHaveBeenClassified = true;
		}

		/// <summary>
		/// Classify the individuals of the model into groups that will be represented in the schema.
		/// </summary>
		/// <remarks>
		/// This method has side effects of filling the following private categories in this model:
		///		mTypesAndMembers
		///		mEnumerationClass
		///		mEnumClassNames
		///		mTypesWithoutMembers
		/// </remarks>
		private void ClassifyIndividuals()
		{
			IEnumerable<OwlIndividual> individuals = from OwlIndividual individual in mIndividuals.Values
													 where (null != individual.MemberOf)
													 //orderby individual.ShortName
													 select individual;
			foreach (OwlIndividual individual in individuals)
			{
				foreach (OwlClass memberOf in individual.MemberOf)
				{
					OwlClass typeClass = IsOrDerivesFromImmediately(memberOf, "Unit");
					if (null == typeClass)
					{
						typeClass = IsOrDerivesFromTwicePenultimately(memberOf, "Unit");
						if (null == typeClass)
						{
							typeClass = IsOrDerivesFromImmediately(memberOf, "QuantityKind");
							if (null == typeClass)
							{
								typeClass = IsOrDerivesFromImmediately(memberOf, "ExtensibleEnumeration");
								if (null == typeClass)
								{
									typeClass = IsOrDerivesFromImmediately(memberOf, "SemanticProperty");
									if (null == typeClass)
									{
										RecordMemberOfType(individual, "NotSemantic", null);
									}
									else
									{
										// The individual's class derives from SemanticProperty, so
										// the individual will be a member of an enumeration in the
										// schema to be generated.
										string typeClassName = typeClass.ShortName;
										if (typeClassName == "enumeration")
										{
											// The individual represents an enumeration data value
											// in an interface.
											mEnumerationClass = typeClass;
											string enumClassName = memberOf.ShortName;
											RecordMemberOfType(individual, enumClassName, memberOf);
											if (false == mEnumClassNames.ContainsKey(enumClassName))
											{
												mEnumClassNames.Add(enumClassName, memberOf);
											}
										}
										else
										{
											// The individual represents an enumeration property of a
											// data value in an interface.
											RecordMemberOfType(individual, typeClassName, typeClass);
										}
									}
								}
								else
								{
									RecordMemberOfType(individual, typeClass.ShortName, typeClass);
								}
							}
							else
							{
								RecordMemberOfType(individual, "QuantityKind", null);
							}
						}
						else
						{
							RecordMemberOfType(individual, "Unit", null);
						}
					}
					else
					{
						RecordMemberOfType(individual, "Unit", null);
					}
				}
			}
		}

		/// <summary>
		/// Identify semantic attributes that are not enumerations.
		/// </summary>
		/// <remarks>
		/// This method has the side effect of adding classes to the private category mTypesWithoutMembers.
		/// </remarks>
		private void IdentifyTypesWithoutMembers()
		{
			IEnumerable<OwlClass> classes = from OwlClass owlClass in mClasses.Values
											where (
											(null != IsOrDerivesFromImmediately(owlClass, "SemanticProperty"))
											&&
											("SemanticProperty" != owlClass.ShortName)
											&&
											("RefersToModel" != owlClass.ShortName)
											&&
											("enumeration" != owlClass.ShortName)
											&&
											(! mTypesAndMembers.ContainsKey(owlClass.ShortName))
											)
											select owlClass;
			foreach (OwlClass owlClass in classes)
			{
				mTypesWithoutMembers.Add(owlClass.ShortName, owlClass);
			}
		}

		/// <summary>
		/// Identify classes that are in the model of operation.
		/// </summary>
		/// <remarks>
		/// This method has the side effect of adding classes to the private category mModelsOfOps.
		/// </remarks>
		private void IdentifyPartsOfModelOfOperations()
		{
			IEnumerable<OwlClass> modOps = from OwlClass owlClass in mClasses.Values
										   where (IsA(owlClass, "ModelOfOperation"))
										   select owlClass;
			foreach (OwlClass owlClass in modOps)
			{
				string typeName = owlClass.ShortName;
				if (false == mModelsOfOps.ContainsKey(typeName))
				{
					mModelsOfOps.Add(typeName, owlClass);
				}
			}
		}

		/// <summary>
		/// Determine whether a class is a particular class, or derives from the particular class.
		/// </summary>
		/// <param name="query">the class that is the subject of the question</param>
		/// <param name="target">the name of the class that may be the same or a base class</param>
		/// <returns>whether the query is the target class or a derivative of it</returns>
		private bool IsA(OwlClass query, string target)
		{
			bool result = false;
			if (null != query)
			{
				if (query.ShortName == target)
				{
					// The query class is the same as the target class.
					result = true;
				}
				else
				{
					if (null != query.SubclassOf)
					{
						foreach (OwlClass item in query.SubclassOf)
						{
							result = IsA(item, target);
							if (result)
							{
								// The target class is an ancestor of the query class.
								break;
							}
						}
					}
				}
			}
			return result;
		}

		/// <summary>
		/// Determine whether a class is a particular class,
		/// or derives from the particular class with no ancestors in between.
		/// </summary>
		/// <param name="query">the class that is the subject of the question</param>
		/// <param name="target">the name of the class that may be the same or an immediate ancestor</param>
		/// <returns>
		/// the query class if it is the target,
		/// the direct subclass of the target class from which the query class derives,
		/// or null if the query class does not derive from the target
		/// </returns>
		private OwlClass IsOrDerivesFromImmediately(OwlClass query, string target)
		{
			OwlClass result = null;
			if (null != query)
			{
				if (query.ShortName == target)
				{
					// The query class is the same as the target class.
					result = query;
				}
				else
				{
					if (null != query.SubclassOf)
					{
						foreach (OwlClass item in query.SubclassOf)
						{
							result = IsOrDerivesFromImmediately(item, target);
							if (null != result)
							{
								// The target class is an ancestor of the query class.
								if (item.ShortName == target)
								{
									// The target class is the immediate ancestor of the query class.
									result = query;
								}
								break;
							}
						}
					}
				}
			}
			return result;
		}

		/// <summary>
		/// Determine whether a class is a particular class, or derives directly from a direct derivative of the particular class.
		/// </summary>
		/// <param name="query">the class that is the subject of the question</param>
		/// <param name="target">the name of the class that may be the same or a base class</param>
		/// <returns>the query class or null if the relationship does not hold</returns>
		private OwlClass IsOrDerivesFromTwicePenultimately(OwlClass query, string target)
		{
			OwlClass result = null;
			if (null == query)
			{
				result = null;
			}
			else
			{
				if (query.ShortName == target)
				{
					// The query class is the same as the target class.
					result = query;
				}
				else
				{
					if (null == query.SubclassOf)
					{
						result = null;
					}
					else
					{
						foreach (OwlClass item in query.SubclassOf)
						{
							result = IsOrDerivesFromImmediately(item, target);
							if (null == result)
							{
								if (null == item.SubclassOf)
								{
									result = null;
								}
								else
								{
									foreach (OwlClass second in item.SubclassOf)
									{
										result = IsOrDerivesFromImmediately(second, target);
										if (null != result)
										{
											// The target class is an ancestor of the query class.
											if (second.ShortName == target)
											{
												// The target class is the grand (second) ancestor of the query class.
												result = query;
											}
											break;
										}
									}
								}
							}
							else
							{
								if (item.ShortName == target)
								{
									result = query;
								}
								break;
							}
						}
					}
				}
			}
			return result;
		}

		/// <summary>
		/// Add an individual to the dictionary of semantic classes with their individuals.
		/// </summary>
		/// <param name="member">an individual to be classified in a semantic class</param>
		/// <param name="typeName">the name of the semantic class</param>
		/// <param name="typeClass">the semantic class</param>
		private void RecordMemberOfType(OwlIndividual member, string typeName, OwlClass typeClass)
		{
			KeyValuePair<OwlClass, List<OwlIndividual>> pair;
			if (false == mTypesAndMembers.ContainsKey(typeName))
			{
				pair = new KeyValuePair<OwlClass, List<OwlIndividual>>(typeClass, new List<OwlIndividual>());
				mTypesAndMembers.Add(typeName, pair);
			}
			else
			{
				pair = mTypesAndMembers[typeName];
			}
			List<OwlIndividual> members = pair.Value;
			members.Add(member);
		}

		/// <summary>
		/// a dictionary of semantic classes with their individuals
		/// </summary>
		Dictionary<string, KeyValuePair<OwlClass, List<OwlIndividual>>> mTypesAndMembers = new Dictionary<string, KeyValuePair<OwlClass, List<OwlIndividual>>>();

		/// <summary>
		/// the class for types of enumerated data that can pass through an interface, or null if none such in model
		/// </summary>
		private OwlClass mEnumerationClass = null;

		/// <summary>
		/// a dictionary of types of enumerated data that can pass through interfaces described by EDS's
		/// </summary>
		private Dictionary<string, OwlClass> mEnumClassNames = new Dictionary<string, OwlClass>();

		/// <summary>
		/// a dictionary of semantic attributes that are not enumerations
		/// </summary>
		/// <remarks>
		/// Referential attributes, such as "subject", take values that are not enumerated.
		/// These attributes take values that are controlled by a regular expression.
		/// </remarks>
		Dictionary<string, OwlClass> mTypesWithoutMembers = new Dictionary<string, OwlClass>();

		/// <summary>
		/// a dictionary of classes that are in the model of operations
		/// </summary>
		Dictionary<string, OwlClass> mModelsOfOps = new Dictionary<string, OwlClass>();

		/// <summary>
		/// a dictionary of simple types constrained by regular expressions
		/// </summary>
		Dictionary<string, OwlIndividual> mSpecificValues = new Dictionary<string, OwlIndividual>();

		/// <summary>
		/// whether the individuals in the model have been classified
		/// </summary>
		private bool mModelPartsHaveBeenClassified = false;

		#endregion

		#region Render this model as html.

        /// <summary>
        /// Render this model as an html file.
        /// </summary>
		public string AsHtml
        {
            get
            {
                ClassifyModelParts();
                string result = "";
                int indent = 0;
                result += "<HTML>";
                NewLineHtml(++indent, result);
                result += "<HEAD>";
                NewLineHtml(++indent, result);
                result += "<TITLE>SOIS DoT</TITLE>";
                NewLineHtml(--indent, result);
                result += "</HEAD>";
                NewLineHtml(indent, result);
                result += "<BODY BGCOLOR=\"WHITE\">";
                NewLineHtml(++indent, result);
                result += "<table border=\"2\">";
                NewLineHtml(++indent, result);
                result += "<tr>";
                NewLineHtml(++indent, result);
                result += "<th>Class Name</th>";
                NewLineHtml(indent, result);
                result += "<th>Base Class</th>";
                NewLineHtml(indent, result);
                result += "<th>Description</th>";
                NewLineHtml(--indent, result);
                result += "</tr>";
                foreach (string typeName in mTypesAndMembers.Keys)
                {
                    if (typeName == "NotSemantic") continue;
                    KeyValuePair<OwlClass, List<OwlIndividual>> pair = mTypesAndMembers[typeName];
                    OwlClass item = pair.Key;
                    if (null != IsOrDerivesFromImmediately(item, "SchemaProperty")) continue;
                    NewLineHtml(indent, result);
                    result += "<tr>";
                    NewLineHtml(++indent, result);
                    result += "<td>" + typeName + "</td>";
                    NewLineHtml(indent, result);
                    result += "<td>";
                    if (null != item)
                    {
                        if (null != item.SubclassOf)
                        {
                            bool additional = false;
                            foreach (OwlClass superclass in item.SubclassOf)
                            {
                                if (additional)
                                {
                                    result += ", ";
                                }
                                result += superclass.DisplayName;
                            }
                        }
                    }
                    result += "</td>";
                    NewLineHtml(indent, result);
                    result += "<td>";
                    if (null != item)
                    {
                        if (null != item.Description)
                        {
                            result += NormalizeLineEndsToWindows(item.Description);
                        }
                    }
                    result += "</td>";
                    NewLineHtml(--indent, result);
                    result += "</tr>";
                }
                NewLineHtml(--indent, result);
                result += "</table>";
                //NewLineHtml(indent, result);
                //result += "<table border=\"2\">";
                //NewLineHtml(++indent, result);
                //result += "<tr>";
                //NewLineHtml(++indent, result);
                //result += "<th>Object Property</th>";
                //NewLineHtml(indent, result);
                //result += "<th>Domain</th>";
                //NewLineHtml(indent, result);
                //result += "<th>Range</th>";
                //NewLineHtml(indent, result);
                //result += "<th>Description</th>";
                //NewLineHtml(--indent, result);
                //result += "</tr>";

                //	<xsl:for-each select="*/owl:ObjectProperty">
                //		<xsl:call-template name="ExtractObjectProperty"/>
                //	</xsl:for-each>

                //NewLineHtml(--indent, result);
                //result += "</table>";
                //NewLineHtml(indent, result);
                //result += "<table border=\"2\">";
                //NewLineHtml(++indent, result);
                //result += "<tr>";
                //NewLineHtml(++indent, result);
                //result += "<th>Datatype Property</th>";
                //NewLineHtml(indent, result);
                //result += "<th>Domain</th>";
                //NewLineHtml(indent, result);
                //result += "<th>Range</th>";
                //NewLineHtml(indent, result);
                //result += "<th>Description</th>";
                //NewLineHtml(--indent, result);
                //result += "</tr>";

                //	<xsl:for-each select="*/owl:DatatypeProperty">
                //		<xsl:call-template name="ExtractDatatypeProperty"/>
                //	</xsl:for-each>

                //NewLineHtml(--indent, result);
                //result += "</table>";
                NewLineHtml(indent, result);
                result += "<table border=\"2\">";
                NewLineHtml(++indent, result);
                result += "<tr>";
                NewLineHtml(++indent, result);
                result += "<th>Instance Name</th>";
                NewLineHtml(indent, result);
                result += "<th>Class</th>";
                NewLineHtml(indent, result);
                result += "<th>Description</th>";
                NewLineHtml(--indent, result);
                result += "</tr>";
                foreach (string typeName in mTypesAndMembers.Keys)
                {
                    if (typeName == "NotSemantic") continue;
                    KeyValuePair<OwlClass, List<OwlIndividual>> pair = mTypesAndMembers[typeName];
                    OwlClass item = pair.Key;
                    if (null != IsOrDerivesFromImmediately(item, "SchemaProperty")) continue;
                    foreach (OwlIndividual member in pair.Value)
                    {
                        NewLineHtml(indent, result);
                        result += "<tr>";
                        NewLineHtml(++indent, result);
                        result += "<td>" + member.ShortName + "</td>";
                        NewLineHtml(indent, result);
                        result += "<td>" + typeName + "</td>";
                        NewLineHtml(indent, result);
                        result += "<td>";
                        if (null != member.Description)
                        {
                            result += NormalizeLineEndsToWindows(member.Description);
                        }
                        result += "</td>";
                        NewLineHtml(--indent, result);
                        result += "</tr>";
                    }
                }
                NewLineHtml(--indent, result);
                result += "</table>";
                NewLineHtml(--indent, result);
                result += "</BODY>";
                NewLineHtml(--indent, result);
                result += "</HTML>";
                System.Diagnostics.Debug.Assert(0 == indent, "indentation error: " + indent.ToString() + ".");
                return result;
            }
        }

        /// <summary>
        /// Extend a string with a new line and tabs to indent the next line.
        /// </summary>
        /// <param name="indent">the number of tabs to indent the next line</param>
        /// <param name="html">the string to be extended</param>
        private void NewLineHtml(int indent, string html)
        {
            html += System.Environment.NewLine;
            for (int ix = 0; ix < indent; ++ix) html += "\t";
        }

        #endregion

        #region the namespaces in which this model participates

        /// <summary>
        /// the name of the attribute for the default namespace
        /// </summary>
        public const string XmlNamespace = "xmlns";

        /// <summary>
        /// the Dublin core namespace
        /// </summary>
        public const string DublinSpace = "http://purl.org/dc/elements/1.1/";

        /// <summary>
        /// the web ontology language (owl) namespace
        /// </summary>
        public const string OwlSpace = "http://www.w3.org/2002/07/owl#";

        /// <summary>
        /// the resource description framework namespace
        /// </summary>
        public const string RdfSpace = "http://www.w3.org/1999/02/22-rdf-syntax-ns#";

        /// <summary>
        /// the resource description framework schema namespace
        /// </summary>
		public const string RdfsSpace = "http://www.w3.org/2000/01/rdf-schema#";

        /// <summary>
        /// the xml schema namespace
        /// </summary>
        public const string XsdSpace = "http://www.w3.org/2001/XMLSchema#";

        /// <summary>
        /// the default namespace in which this model is defined
        /// </summary>
        /// <remarks>
        /// This namespace is identified by the xmlns attribute (without prefix) in xml.
        /// </remarks>
        public string DefaultNamespace { get; set; }

        /// <summary>
        /// Assure that a given namespace name is known in this model.
        /// </summary>
        /// <param name="name">the name of the namespace</param>
        /// <remarks>
        /// This method has the side effect of adding a namespace to the private category mNamespaces.
        /// </remarks>
        public void FindOrMakeNamespace(string name)
        {
            if (!mNamespaces.ContainsKey(name))
            {
                mNamespaces.Add(name, name);
            }
        }

        /// <summary>
        /// Given the trunk element of an xml document object model (DOM), determine the default namespace for the DOM.
        /// </summary>
        /// <param name="trunk">the trunk element of the xml document object model whose default namespace is wanted</param>
        /// <remarks>
        /// "rdf:RDF xmlns" provides namespace name, stored in mNamespaces dictionary.
        /// The default namespace becomes the value of the DefaultNamespace property of this model.
        /// </remarks>
        private void DetermineDefaultNamespace(XElement trunk)
        {
            if (XmlHelp.ElementIs(trunk, RdfSpace, Rdf))
            {
                XAttribute namespaceAttribute = trunk.Attribute(XmlNamespace);
                if (null == namespaceAttribute)
                {
                    DefaultNamespace = "";
                }
                else
                {
                    DefaultNamespace = OwlNameInSpace.NormalizeNamespace(namespaceAttribute.Value);
                }
                FindOrMakeNamespace(DefaultNamespace);
            }
            else
            {
                throw new Exception("trunk element is " + trunk.Name + ", which is not supported by this program; " + Rdf + " is supported.");
            }
        }

        /// <summary>
        /// The catalog of name spaces is a sorted list of name space names.
        /// </summary>
        /// <remarks>
        /// The values are the same as the keys.
        /// </remarks>
        private Dictionary<string, string> mNamespaces = new Dictionary<string, string>();

        #endregion

        #region the data types in this model

        /// <summary>
        /// Assure that a given datatype name is known in this model.
        /// </summary>
        /// <param name="name">the name of the datatype</param>
        /// <remarks>
        /// This method has the side effect of adding a datatype to the private category mDatatypes.
        /// </remarks>
        public Datatype FindOrMakeDatatype(string name)
		{
			Datatype result = null;
			if (mDatatypes.ContainsKey(name))
			{
				result = mDatatypes[name];
			}
			else
			{
				result = new Datatype();
				result.InModel = this;
				result.About = name;
				mDatatypes.Add(name, result);
			}
			return result;
		}

        /// <summary>
        /// The catalog of datatypes is a dictionary of datatype names.
        /// </summary>
        private Dictionary<string, Datatype> mDatatypes = new Dictionary<string, Datatype>();

        #endregion

        #region the properties in this model

        /// <summary>
        /// Assure that a given property name is known in this model.
        /// </summary>
        /// <param name="name">the name of the property</param>
        /// <remarks>
        /// This method has the side effect of adding a namespace to the private category mProperties.
        /// </remarks>
        public Property FindOrMakeProperty(string name)
        {
            Property result = null;
			if (mObjectProperties.ContainsKey(name))
			{
				result = mObjectProperties[name];
			}
            else
            {
				if (mDatatypeProperties.ContainsKey(name))
				{
					result = mDatatypeProperties[name];
				}
                else
                {
					if (mProperties.ContainsKey(name))
					{
						result = mProperties[name];
					}
					else
					{
						result = new Property();
						result.InModel = this;
						result.About = name;
						mProperties.Add(name, result);
					}
				}
            }
            return result;
        }

        /// <summary>
        /// The catalog of properties is a dictionary of property names.
        /// </summary>
        private Dictionary<string, Property> mProperties = new Dictionary<string, Property>();

        #endregion

		#region the object properties in this model

		/// <summary>
		/// Given the name of an object property, return the object that represents that object property.
		/// </summary>
		/// <param name="name">the name of the object property that is wanted</param>
		/// <returns>the object that represents the object property with the specified name</returns>
		/// <remarks>
		/// This method can have the side effect of adding an object property to the private dictionary mObjectProperties.
		/// </remarks>
		public ObjectProperty FindOrMakeObjectProperty(string name)
		{
			ObjectProperty result = null;
			if (mObjectProperties.ContainsKey(name))
			{
				result = mObjectProperties[name];
			}
			else
			{
				result = new ObjectProperty();
				result.InModel = this;
				result.About = name;
				mObjectProperties.Add(name, result);
			}
			return result;
		}

		/// <summary>
		/// Update the dictionary of object properties to agree with a change in the name of an object property.
		/// </summary>
		/// <param name="previouslyKnownAs">the previous name of the affected object property</param>
		/// <param name="renamed">the affected object property</param>
		public void ObjectPropertyChangedName(string previouslyKnownAs, ObjectProperty renamed)
		{
			if (previouslyKnownAs != renamed.About)
			{
				ObjectProperty affected = mObjectProperties[previouslyKnownAs];
				if (null != affected)
				{
					mObjectProperties.Remove(previouslyKnownAs);
				}
				mObjectProperties.Add(renamed.About, renamed);
			}
		}

		/// <summary>
		/// the object properties in this model
		/// </summary>
		private Dictionary<string, ObjectProperty> mObjectProperties = new Dictionary<string, ObjectProperty>();

		#endregion

		#region the datatype properties in this model

		/// <summary>
		/// Given the name of a datatype property, return the object that represents that datatype property.
		/// </summary>
		/// <param name="name">the name of the datatype property that is wanted</param>
		/// <returns>the object that represents the datatype property with the specified name</returns>
		/// <remarks>
		/// This method can have the side effect of adding a datatype property to the private dictionary mDatatypeProperties.
		/// </remarks>
		public DatatypeProperty FindOrMakeDatatypeProperty(string name)
		{
			DatatypeProperty result = null;
			if (mDatatypeProperties.ContainsKey(name))
			{
				result = mDatatypeProperties[name];
			}
			else
			{
				result = new DatatypeProperty();
				result.InModel = this;
				result.About = name;
				mDatatypeProperties.Add(name, result);
			}
			return result;
		}

		/// <summary>
		/// Update the dictionary of datatype properties to agree with a change in the name of a datatype property.
		/// </summary>
		/// <param name="previouslyKnownAs">the previous name of the affected datatype property</param>
		/// <param name="renamed">the affected datatype property</param>
		public void DatatypePropertyChangedName(string previouslyKnownAs, DatatypeProperty renamed)
		{
			if (previouslyKnownAs != renamed.About)
			{
				DatatypeProperty affected = mDatatypeProperties[previouslyKnownAs];
				if (null != affected)
				{
					mDatatypeProperties.Remove(previouslyKnownAs);
				}
				mDatatypeProperties.Add(renamed.About, renamed);
			}
		}

		/// <summary>
		/// the datatype properties in this model
		/// </summary>
		private Dictionary<string, DatatypeProperty> mDatatypeProperties = new Dictionary<string, DatatypeProperty>();

		#endregion

		#region the classes in this model

		/// <summary>
		/// Given the name of a class, return the object that represents that class.
		/// </summary>
		/// <param name="name">the name of the class that is wanted</param>
		/// <returns>the object that represents the class with the specified name</returns>
		/// <remarks>
		/// This method can have the side effect of adding a class to the private dictionary mClasses.
		/// </remarks>
		public OwlClass FindOrMakeClass(string name)
		{
			OwlClass result = null;
			if (mClasses.ContainsKey(name))
			{
				result = mClasses[name];
			}
			else
			{
				result = new OwlClass();
				result.InModel = this;
				result.About = name;
				mClasses.Add(name, result);
			}
			return result;
		}

		/// <summary>
		/// Make an object that represents a new anonymous class.
		/// </summary>
		/// <returns>an object that represents a new anonymous class</returns>
		/// <remarks>
		/// This method has the side effect of adding a class to the private dictionary mClasses.
		/// The new class has an arbitrary name, which distinguishes it from other anonymous classes.
		/// </remarks>
		public OwlClass MakeAnonymousClass()
		{
			OwlClass result = null;
			++mAnonymousClassCount;
			string name = mAnonymousClassCount.ToString();
			result = FindOrMakeClass(name);
			result.Anonymous = true;
			return result;
		}

		/// <summary>
		/// Update the dictionary of classes to agree with a change in the name of a class.
		/// </summary>
		/// <param name="previouslyKnownAs">the previous name of the affected class</param>
		/// <param name="renamed">the affected class</param>
		public void ClassChangedName(string previouslyKnownAs, OwlClass renamed)
		{
			if (previouslyKnownAs != renamed.About)
			{
				OwlClass affected = mClasses[previouslyKnownAs];
				if (null != affected)
				{
					mClasses.Remove(previouslyKnownAs);
				}
				mClasses.Add(renamed.About, renamed);
			}
		}

		/// <summary>
		/// the classes in this model
		/// </summary>
		private Dictionary<string, OwlClass> mClasses = new Dictionary<string, OwlClass>();

		/// <summary>
		/// the number of anonymous classes in this model
		/// </summary>
		/// <remarks>
		/// This integer is used to construct the names of anonymous classes.
		/// </remarks>
		private uint mAnonymousClassCount = 0;

		#endregion

		#region the individuals in this model

		/// <summary>
		/// Given the name of an individual, return the object that represents that individual.
		/// </summary>
		/// <param name="name">the name of the individual that is wanted</param>
		/// <returns>the object that represents the individual with the specified name</returns>
		/// <remarks>
		/// This method can have the side effect of adding an individual to the private dictionary mIndividuals.
		/// </remarks>
		public OwlIndividual FindOrMakeIndividual(string name)
		{
			OwlIndividual result = null;
			if (mIndividuals.ContainsKey(name))
			{
				result = mIndividuals[name];
			}
			else
			{
				result = new OwlIndividual();
				result.InModel = this;
				result.About = name;
				mIndividuals.Add(name, result);
			}
			return result;
		}

		/// <summary>
		/// Make an object that represents a new anonymous individual.
		/// </summary>
		/// <returns>an object that represents a new anonymous individual</returns>
		/// <remarks>
		/// This method has the side effect of adding an individual to the private dictionary mIndividuals.
		/// The new individual has an arbitrary name, which distinguishes it from other anonymous individuals.
		/// </remarks>
		public OwlIndividual MakeAnonymousIndividual()
		{
			OwlIndividual result = null;
			++mAnonymousIndividualCount;
			string name = mAnonymousIndividualCount.ToString();
			result = FindOrMakeIndividual(name);
			result.Anonymous = true;
			return result;
		}

		/// <summary>
		/// Update the dictionary of individuals to agree with a change in the name of an individual.
		/// </summary>
		/// <param name="previouslyKnownAs">the previous name of the affected individual</param>
		/// <param name="renamed">the affected individual</param>
		public void IndividualChangedName(string previouslyKnownAs, OwlIndividual renamed)
		{
			if (previouslyKnownAs != renamed.About)
			{
				OwlIndividual affected = mIndividuals[previouslyKnownAs];
				if (null != affected)
				{
					mIndividuals.Remove(previouslyKnownAs);
				}
				mIndividuals.Add(renamed.About, renamed);
			}
		}

		/// <summary>
		/// the individuals in this model
		/// </summary>
		private Dictionary<string, OwlIndividual> mIndividuals = new Dictionary<string, OwlIndividual>();

		/// <summary>
		/// the number of anonymous individuals in this model
		/// </summary>
		/// <remarks>
		/// This integer is used to construct the names of anonymous individuals.
		/// </remarks>
		private uint mAnonymousIndividualCount = 0;

		#endregion

		#region methods for handling text files consistently

		/// <summary>
		/// Given a string with potentially some line ends not in the windows style,
		/// construct a new version of the string with all line ends normal for windows.
		/// </summary>
		/// <param name="abbie">a string with potentially some line ends not in the windows style</param>
		/// <returns>a new version of the string with all line ends normal for windows</returns>
		private string NormalizeLineEndsToWindows(string abbie)
		{
			string result = abbie.Replace("\r\n", "\n").Replace("\n", "\r\n");
			return result;
		}

		#endregion

	}
}
